%% MOTORMain.m
%% N. Boizot 06 / 2022


%% Attention, la prise de données / l'envoi du controle n'est uniforme 
%% que dans leur presentation dans le fichier en de mesures, pour simplifier 
%% la manipulation des mesures au prix d'une (petite) erreur.

warning off
close all
clear variables
clc


% --------------------------- %
% ---   PARAMETERS   -------- %

% Baudrate = 19200 ; % Must match the Baudrate defined in the arduino program.
Baudrate = 115200 ; % Must match the Baudrate defined in the arduino program.
MOTEUR.Data_Size = 10;
MOTEUR.Encoder_Resolution = 48*172; %% 172:1
% --------------------------- %

if exist('MotorSerial') 
    clear MotorSerial
end

Slist = serialportlist('available');

% ------------ %

% Windows OS
% Appropriate Com port is selected within a list of available ports
Selected = menu('Selectionner le port',Slist);
MotorSerial = serialport(Slist(Selected),Baudrate);  % Or MotorSerial = serialport(Slist(Selected),'19200,n,8,1'); 


disp(Slist(Selected)+' is connected');

% ---------------------------- %
% SELECT Open/Closed Loop Mode %
% ---------------------------- %

Mode = menu('Boucle ouverte ou boucle fermée','Boucle ouverte','Boucle fermée')
% 1 -> ouverte
% 2 -> fermée

% ------ %
% Interprets the Mode variable and forms the appropriate frame that will trigger open/closed loop behavior
% ------ %
Trame = MotorInterpret(Mode,MOTEUR);  

% ------ %

disp('Begin experiment');

% ------ %
% Triggers open/closed loop behavior (i.e. sends the frame)
% waits (6 sec)
% reads and process the data frame provided by Arduino
Tab = MotorMeasure(Trame,MotorSerial);  
% ------ %

disp('End experiment');

%     
disp(Slist(Selected)+' is now closed');

clear MotorSerial
%     
%     
%     
Data = MotorCalcul(Tab,MOTEUR);
save ('Measure.txt', 'Data', '-ascii')
xlswrite('Measure',Data)


figure(1)
subplot(311)
plot(Data(:,1),Data(:,3),'LineWidth',1,'color', 'blue');
xlabel('temps (ms)', 'fontsize', 12, 'color', 'blue');
ylabel('position (°)', 'fontsize', 12, 'color', 'blue');

subplot(312)
plot(Data(:,1),Data(:,4),'LineWidth',1,'Color','blue');
xlabel('temps (ms)', 'fontsize', 12, 'color', 'blue');
ylabel('vitesse (tr/min)', 'fontsize', 12, 'color', 'blue');

subplot(313)
plot(Data(:,1),Data(:,2),'LineWidth',1,'Color','blue');
xlabel('temps (ms)', 'fontsize', 12, 'color', 'blue');
ylabel('u(t)', 'fontsize', 12, 'color', 'blue');


%
%
%
%
% ---------------------------------- %
% ------ Functions definitions ----- %
% (Possible since R2016)

    
    
function Message = MotorInterpret(Mode,MOTEUR)
    Boucle = Mode-1;
    switch Boucle
        case 0
            PPM = inputdlg('u(t) in [-400;400]','Amplitude de l''échelon d''entrée',[1,35],{'0'});
            if length(PPM) == 0, error('Opération interrompue'), end
            PPM = cell2mat(PPM);
            PPM = 500 + floor(str2num(PPM));
            PPM = min(900,max(100,floor(PPM)));
            Message = [num2str(Boucle),num2str(PPM),'00000000'];
        case 1
            answer = inputdlg({'Consigne d''angle','Gain'},'Réglages boucle fermée',[1,35],{'0','0'});
                % ----- %
            Angle = str2num(cell2mat(answer(1)));
            Angle2Ticks = round(Angle * MOTEUR.Encoder_Resolution/360);
            Angle_Offset = 1e6;  % Should be used in Arduino code
            % This centers the Setpoint (consigne) around 1e6
            Code = Angle_Offset+Angle2Ticks;
            Message_Code = num2str(Code,'%07d');
                % ----- %
            G = str2num(cell2mat(answer(2)));
            Gain = round(G * 360 / MOTEUR.Encoder_Resolution*1000);
            Message_Gain = num2str(Gain,'%05d');
            Message = [num2str(Boucle), Message_Code, Message_Gain];     
    end
    disp(Message);
end
    
% %    
        
function Data = MotorMeasure(Message,MotorSerial)
    write(MotorSerial,Message,'uint8');
    XP = tic;
    while toc(XP)<6
        % empty while loop that waits 6 sec.
    end;
    Data_Received = read(MotorSerial,MotorSerial.NumBytesAvailable,'string');
    if length(Data_Received) == 0 then
        warning('There''s no data to read');
        Data = 0;
    else
        Data_Separated = strsplit(Data_Received, '|');
        Data_Separated = Data_Separated(1:end-1);
        Data_Separated = split(Data_Separated,'/');
        Data = double(Data_Separated);   
        Data = squeeze(Data);


      % BootStrap code to make the time intervals uniform 
      % This is cheating in some sense
       Time = Data(:,1);
       PWM = Data(:,2);
       Code = Data(:,3);
       step = round(mean(diff(Time)));
       nTime = (0 : step : Time(end))';
       nCode = interp1(Time,Code,nTime);
       nPWM = interp1(Time,PWM,nTime);
       Data = [nTime,nPWM,nCode];
    end
end
     

function Data_converted = MotorCalcul(Data,MOTEUR)
    Time = Data(:,1)/1e3;;
    Input = Data(:,2)-500;
    Position = (Data(:,3)-2^20)*360/MOTEUR.Encoder_Resolution;

    Variation_Position = diff(Position);
    Variation_Time =diff(Time);
    
    
%    Speed = Variation_Position*60000./(Variation_Time*360);
    Speed = Variation_Position*60000./(Variation_Time*360);
    Speed = [0;Speed];

    Data_converted = [Time,Input,Position,Speed];   
    
    Data_converted(isnan(Data_converted))=0;
end
